'use strict';

/* --------------------------------------------------------------
 datatable.js 2016-07-11
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.libs.datatable = jse.libs.datatable || {};

/**
 * ## DataTable Library
 *
 * This is a wrapper library for the manipulation of jQuery DataTables. Use the "create" method with DataTable
 * configuration to initialize a table on your page. All you need when using this library is an empty `<table>`
 * element. Visit the official website of DataTables to check examples and other information about the plugin.
 *
 * {@link http://www.datatables.net Official DataTables Website}
 *
 * Notice: Make sure that you load the DataTables vendor files before using this module.
 *
 * ### Examples
 *
 * **Example - Create A New Instance**
 * ```javascript
 * var tableApi = jse.libs.datatable.create($('#my-table'), {
 *      ajax: 'http://shop.de/table-data.php',
 *      columns: [
 *          { title: 'Name', data: 'name' defaultContent: '...' },
 *          { title: 'Email', data: 'email' },
 *          { title: 'Actions', data: null, orderable: false, defaultContent: 'Add | Edit | Delete' },
 *      ]
 * });
 * ```
 *
 * **Example - Add Error Handler**
 * ```javascript
 * jse.libs.datatable.error($('#my-table'), function(event, settings, techNote, message) {
 *      // Log error in the JavaScript console.
 *      console.log('DataTable Error:', message);
 * });
 * ```
 *
 * @module JSE/Libs/datatable
 * @exports jse.libs.datatable
 * @requires jQuery-DataTables-Plugin
 */
(function (exports) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    var languages = {
        de: {
            'sEmptyTable': 'Keine Daten in der Tabelle vorhanden',
            'sInfo': '_START_ bis _END_ (von _TOTAL_)',
            'sInfoEmpty': '0 bis 0 von 0 Einträgen',
            'sInfoFiltered': '(gefiltert von _MAX_ Einträgen)',
            'sInfoPostFix': '',
            'sInfoThousands': '.',
            'sLengthMenu': '_MENU_ Einträge anzeigen',
            'sLoadingRecords': 'Wird geladen...',
            'sProcessing': 'Bitte warten...',
            'sSearch': 'Suchen',
            'sZeroRecords': 'Keine Einträge vorhanden.',
            'oPaginate': {
                'sFirst': 'Erste',
                'sPrevious': 'Zurück',
                'sNext': 'Nächste',
                'sLast': 'Letzte'
            },
            'oAria': {
                'sSortAscending': ': aktivieren, um Spalte aufsteigend zu sortieren',
                'sSortDescending': ': aktivieren, um Spalte absteigend zu sortieren'
            }
        },
        en: {
            'sEmptyTable': 'No data available in table',
            'sInfo': '_START_ to _END_ (of _TOTAL_)',
            'sInfoEmpty': 'Showing 0 to 0 of 0 entries',
            'sInfoFiltered': '(filtered from _MAX_ total entries)',
            'sInfoPostFix': '',
            'sInfoThousands': ',',
            'sLengthMenu': 'Show _MENU_ entries',
            'sLoadingRecords': 'Loading...',
            'sProcessing': 'Processing...',
            'sSearch': 'Search:',
            'sZeroRecords': 'No matching records found',
            'oPaginate': {
                'sFirst': 'First',
                'sLast': 'Last',
                'sNext': 'Next',
                'sPrevious': 'Previous'
            },
            'oAria': {
                'sSortAscending': ': activate to sort column ascending',
                'sSortDescending': ': activate to sort column descending'
            }
        }
    };

    // ------------------------------------------------------------------------
    // FUNCTIONALITY
    // ------------------------------------------------------------------------

    /**
     * Reorder the table columns as defined in the active columns array.
     *
     * @param {jQuery} $target Table jQuery selector object.
     * @param {Object} columnDefinitions Array containing the DataTable column definitions.
     * @param {Array} activeColumnNames Array containing the slug-names of the active columns.
     *
     * @return {Array} Returns array with the active column definitions ready to use in DataTable.columns option.
     *
     * @private
     */
    function _reorderColumns($target, columnDefinitions, activeColumnNames) {
        activeColumnNames.unshift('checkbox');
        activeColumnNames.push('actions');

        // Hide the table header cells that are not active.
        $.each(columnDefinitions, function (index, columnDefinition) {
            $target.find('thead tr').each(function () {
                var $headerCell = $(this).find('[data-column-name="' + columnDefinition.name + '"]');

                if (columnDefinition.data !== null && activeColumnNames.indexOf(columnDefinition.name) === -1) {
                    $headerCell.hide();
                }
            });
        });

        // Prepare the active column definitions.
        var finalColumnDefinitions = [],
            columnIndexes = [];

        $.each(activeColumnNames, function (index, name) {
            $.each(columnDefinitions, function (index, columnDefinition) {
                if (columnDefinition.name === name) {
                    // Add the active column definition in the "finalColumnDefinitions" array.
                    finalColumnDefinitions.push(columnDefinition);
                    var headerCellIndex = $target.find('thead:first tr:first [data-column-name="' + columnDefinition.name + '"]').index();
                    columnIndexes.push(headerCellIndex);
                    return true; // continue
                }
            });
        });

        finalColumnDefinitions.sort(function (a, b) {
            var aIndex = activeColumnNames.indexOf(a.name);
            var bIndex = activeColumnNames.indexOf(b.name);

            if (aIndex < bIndex) {
                return -1;
            } else if (aIndex > bIndex) {
                return 1;
            } else {
                return 0;
            }
        });

        // Reorder the table header elements depending the activeColumnNames order.
        $target.find('thead tr').each(function () {
            var _this = this;

            var activeColumnSelections = [$(this).find('th:first')];

            // Sort the columns in the correct order.
            columnIndexes.forEach(function (index) {
                var $headerCell = $(_this).find('th').eq(index);
                activeColumnSelections.push($headerCell);
            });

            // Move the columns to their final position.
            activeColumnSelections.forEach(function ($headerCell, index) {
                if (index === 0) {
                    return true;
                }

                $headerCell.insertAfter(activeColumnSelections[index - 1]);
            });
        });

        return finalColumnDefinitions;
    }

    /**
     * Creates a DataTable Instance
     *
     * This method will create a new instance of datatable into a `<table>` element. It enables
     * developers to easily pass the configuration needed for different and more special situations.
     *
     * @param {jQuery} $target jQuery object for the target table.
     * @param {Object} configuration DataTables configuration applied on the new instance.
     *
     * @return {DataTable} Returns the DataTable API instance (different from the jQuery object).
     */
    exports.create = function ($target, configuration) {
        return $target.DataTable(configuration);
    };

    /**
     * Sets the error handler for specific DataTable.
     *
     * DataTables provide a useful mechanism that enables developers to control errors during data parsing.
     * If there is an error in the AJAX response or some data are invalid in the JavaScript code you can use
     * this method to control the behavior of the app and show or log the error messages.
     *
     * {@link http://datatables.net/reference/event/error}
     *
     * @param {jQuery} $target jQuery object for the target table.
     * @param {Object} callback Provide a callback method called with the "event", "settings", "techNote",
     * "message" arguments (see provided link).
     */
    exports.error = function ($target, callback) {
        $.fn.dataTable.ext.errMode = 'none';
        $target.on('error.dt', callback).on('xhr.dt', function (event, settings, json, xhr) {
            if (json.exception === true) {
                callback(event, settings, null, json.message);
            }
        });
    };

    /**
     * Sets the callback method when ajax load of data is complete.
     *
     * This method is useful for checking PHP errors or modifying the data before
     * they are displayed to the server.
     *
     * {@link http://datatables.net/reference/event/xhr}
     *
     * @param {jQuery} $target jQuery object for the target table.
     * @param {Function} callback Provide a callback method called with the "event", "settings", "techNote",
     * "message" arguments (see provided link).
     */
    exports.ajaxComplete = function ($target, callback) {
        $target.on('xhr.dt', callback);
    };

    /**
     * Sets the table column to be displayed as an index.
     *
     * This method will easily enable you to set a column as an index column, used
     * for numbering the table rows regardless of the search, sorting and row count.
     *
     * {@link http://www.datatables.net/examples/api/counter_columns.html}
     *
     * @param {jQuery} $target jQuery object for the target table.
     * @param {Number} columnIndex Zero based index of the column to be indexed.
     */
    exports.indexColumn = function ($target, columnIndex) {
        $target.on('order.dt search.dt', function () {
            $target.DataTable().column(columnIndex, {
                search: 'applied',
                order: 'applied'
            }).nodes().each(function (cell, index) {
                cell.innerHTML = index + 1;
            });
        });
    };

    /**
     * Returns the german translation of the DataTables
     *
     * This method provides a quick way to get the language JSON without having to perform
     * and AJAX request to the server. If you setup your DataTable manually you can set the
     * "language" attribute with this method.
     *
     * @deprecated Since v1.4, use the "getTranslations" method instead.
     *
     * @return {Object} Returns the german translation, must be the same as the "german.lang.json" file.
     */
    exports.getGermanTranslation = function () {
        jse.core.debug.warn('DataTables Library: the getGermanTranslation method is deprecated and will be removed ' + 'in JSE v1.5, please use the "getTranslations" method instead.');
        return languages.de;
    };

    /**
     * Get the DataTables translation depending the language code parameter.
     *
     * @param {String} languageCode Provide 'de' or 'en' (you can also use the jse.core.config.get('languageCode') to
     * get the current language code).
     *
     * @return {Object} Returns the translation strings in an object literal as described by the official DataTables
     * documentation.
     *
     * {@link https://www.datatables.net/plug-ins/i18n}
     */
    exports.getTranslations = function (languageCode) {
        if (languages[languageCode] === undefined) {
            jse.core.debug.warn('DataTables Library: The requested DataTables translation was not found:', languageCode);
            languageCode = 'en';
        }

        return languages[languageCode];
    };

    /**
     * Prepare table columns.
     *
     * This method will convert the column definitions to a DataTable compatible format and also reorder
     * the table header cells of the "thead" element.
     *
     * @param {jQuery} $target Table jQuery selector object.
     * @param {Object} columnDefinitions Array containing the DataTable column definitions.
     * @param {String[]} activeColumnNames Array containing the slug-names of the active columns.
     *
     * @return {Object[]} Returns array with the active column definitions ready to use in DataTable.columns option.
     */
    exports.prepareColumns = function ($target, columnDefinitions, activeColumnNames) {
        var convertedColumnDefinitions = [];

        for (var columnName in columnDefinitions) {
            var columnDefinition = columnDefinitions[columnName];
            columnDefinition.name = columnName;
            convertedColumnDefinitions.push(columnDefinition);
        }

        return _reorderColumns($target, convertedColumnDefinitions, activeColumnNames);
    };
})(jse.libs.datatable);
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
